﻿/*!
\file csvtable.js
解析csv表格为对象数组

\author jiangyong
\update
2022/7/15 v1.0.1 增加直接转为二维对象表
2022/7/10 v1.0.0 
*/

/**
 * CsvTable对象，用于dbman
 * @param {number} keyrow 列名所在行号(从0开始)
 * @param {function} onCsvKeyVal 字段值预处理
 */
function CsvTable(keyrow = 0, onCsvKeyVal = null) {
	this.keyrow = keyrow;//列名行,默认第一行,数据从keyrow+1行开始
	this.keymap = new Map(); //列名和主见的对照表 key 为列名，val = 对象字段名,用于列名和对象字段名不一致,如果不存在，则实用列名
	this.records = new Array();// 解析后存放的结果集
	this.keys = new Array(); // 以列号为下表的数组

	/**
	* 添加列名和属性名对照
	* @param {string} strColname 列名
	* @param {string} strKey 对象属性名
	*/
	this.addColObjKey = function (strColname, strKey) {
		this.keymap.set(strColname, strKey);
	}

	/**
	* 解析csv字符串
	* @param {string} strcsv 字符串,已经是utf8编码
	*/
	this.parseCsvString = function (strcsv) {
		let i = 0, n = strcsv.length, nline = 0, ncol = 0;
		let nstr = 0;//引号匹配
		let ch = 0, posp = 0,cp=0;
		let objrec = {};
		for (i = 0; i < n; i++) {
			ch = strcsv.charAt(i);
			switch (ch) {
				case ',':
					if ((nstr % 2) !== 0) {
						break;
					}
					if (nline == this.keyrow) {
						if (i > posp)
							this.addCsvColName(strcsv.substr(posp, i - posp), ncol);
						else
							this.addCsvColName("", ncol);
					}
					else if (nline > this.keyrow) {
						if (i > posp)
							this.addCsvField(strcsv.substr(posp, i - posp), ncol, objrec);
					}
					ncol++;
					posp = i + 1;
					break;
				case '\n':
				case '\r':
					if (nline == this.keyrow) {
						if (i > posp) {
							this.addCsvColName(strcsv.substr(posp, i - posp), ncol);
							nline++;
						}
						else if (cp == ',') {
							this.addCsvColName("", ncol);
							nline++;
						}
					}
					else if (nline > this.keyrow) {
						if (i > posp)
							this.addCsvField(strcsv.substr(posp, i - posp), ncol, objrec);
						if (!this.objempty(objrec)) {
							this.records.push(objrec);
							nline++;
							objrec = {};
						}
					}
					else {
						if (i > posp || cp == ',') {
							nline++;
						}
					}
					ncol = 0;
					posp = i + 1;
					break;
				case '"':
					if (i + 1 == n)
						break;
					if ('"' == strcsv.charAt(i + 1)) {
						i++;
					}
					else {
						nstr++;
					}
					break;
			}
			cp = ch;
		}
		if (nline > this.keyrow) {
			if (i > posp)
				this.addCsvField(strcsv.substr(posp, i - posp), ncol, objrec);
			if (!this.objempty(objrec)) {
				this.records.push(objrec);
				objrec = {};
			}
		}
	}

	/**
	* 
	* @param {string} strColname 列名
	* @param {number} numCol 列号
	*/
	this.addCsvColName = function (strColname, numCol) {
		let cv = {};
		if (undefined == strColname || "" == strColname)
			cv.colname = "col" + numCol;
		else
			cv.colname = strColname;
		let kv = this.keymap.get(strColname);
		if (undefined == kv)
			cv.keyname = cv.colname;
		else
			cv.keyname = kv;
		this.keys[numCol] = cv;
	};


	/**
	 * 添加字段值到标签属性
	 * @param {string} sv 字段值字符串,未转码
	 * @param {number} numcol 列号
	 * @param {object} objrec 输出对象
	 */
	this.addCsvField = function (sv, numcol, objrec) {
		if (numcol < 0)
			return;
		let trans = false;
		for (const c of sv) {
			if (c == '"') {
				trans = true;
				break;
			}
		}

		if (trans) { //需要转码
			let s = "";
			let i = 0, n = sv.length;
			let ch = 0;
			for (i = 0; i < n; i++) {
				ch = sv.charAt(i);
				if ('\"' == ch) {
					if (i + 1 == n)
						break;
					if ('\"' == sv.charAt(i + 1)) {
						s += '\"';
						i++;
					}
				}
				else
					s += ch;
			}
			sv = s;
		}

		let kv = this.keys[numcol];
		if (undefined == kv || kv == null) {
			return;
		}
		if (!this.onCsvKeyVal)
			objrec[kv.keyname] = sv;
		else
			objrec[kv.keyname] = this.onCsvKeyVal(kv.colname, sv);
	};

	this.onCsvKeyVal = onCsvKeyVal; //字段预处理,外部重载，不重载默认直接写入
	this.objempty = function (obj) {
		for (let i in obj) {
			return false;
		}
		return true;
	}
	this.clear = function () {
		this.keymap.clear();
		this.records.length = 0;
		this.keys.length = 0;
	}
}


/**
 * CsvRowCols对象，用于IOServer
 * @param {number} keyrow 列名所在行号(从0开始)
 * @param {function} onCsvKeyVal 字段值预处理
 */
function CsvRowCols() {
	this.records = new Array();// 解析后存放的结果集
	this.numcols = 0;//字段数
	this.titlerow = 0;//标题行
	this.clear = function () {
		this.records.length = 0;
		this.numcols = 0;
	}

	/**
	 * 添加字段值到标签属性
	 * @param {string} sv 字段值字符串,未转码
	 * @param {object} array_row 行数组
	 */
	this.addField = function (sv, array_row) {
		let trans = false;
		for (const c of sv) {
			if (c == '"') {
				trans = true;
				break;
			}
		}
		if (trans) { //需要转码
			let s = "";
			let i = 0, n = sv.length;
			let ch = 0;
			for (i = 0; i < n; i++) {
				ch = sv.charAt(i);
				if ('\"' == ch) {
					if (i + 1 == n)
						break;
					if ('\"' == sv.charAt(i + 1)) {
						s += '\"';
						i++;
					}
				}
				else
					s += ch;
			}
			sv = s;
		}
		array_row.push(sv);
	}

	/**
	* 解析csv字符串
	* @param {string} strcsv 字符串,已经是utf8编码
	*/
	this.fromString = function (strcsv) {
		let i = 0, n = strcsv.length, nline = 0, ncol = 0;
		let nstr = 0;//引号匹配
		let ch = 0, posp = 0;
		let temprow = new Array();
		this.numcols = 0;
		for (i = 0; i < n; i++) {
			ch = strcsv.charAt(i);
			switch (ch) {
				case ',':
					if ((nstr % 2) !== 0) {
						break;
					}
					this.addField(strcsv.substr(posp, i - posp), temprow);
					ncol++;

					posp = i + 1;
					break;
				case '\n':
				case '\r':
					if (i > posp) {
						this.addField(strcsv.substr(posp, i - posp), temprow);
						ncol++;
					}
					if (temprow.length > 0) {
						this.records.push(temprow);
						temprow = new Array();
						nline++;
					}
					if (ncol > this.numcols)
						this.numcols = ncol;
					ncol = 0;
					posp = i + 1;
					break;
				case '"':
					if (i + 1 == n)
						break;
					if ('"' == strcsv.charAt(i + 1)) {
						i++;
					}
					else {
						nstr++;
					}
					break;
			}
		}
		if (i > posp) {
			this.addField(strcsv.substr(posp, i - posp), temprow);
			ncol++;
		}
		if (temprow.length > 0) {
			this.records.push(temprow);
			temprow = new Array();
		}
		if (ncol > this.numcols)
			this.numcols = ncol;
		if (this.records.length > 1 && this.records[1].length > this.records[0].length)
			this.titlerow = 1;
	}
}

