/*@
 * 实时库WS/WSS协议通讯JNI接口
 * @author 蒋勇
 * @data: 2021-03-15
 */
package kipway;

/*
 * jniwsapi是没有成员变量的静态Jni接口类,用一个大于等于0的整数来表示一个连接句柄，类似linux里
 * 的fd句柄。句柄可以保存下来多次使用。连接是自动重连的，使用getStatus获取状态。
 * 连接句柄不保证线程安全，建议每个线程Create一个句柄，每个进程最多可以同时并存64个句柄。
 * 不再使用的句柄或者程序退出时必须像文件IO流对象一样显式的调用Close关闭并释放有Create创建的句
 * 柄，以释放线程和内存资源，否则会造成内存泄露。
 */
public class jniwsapi {
	public final static int ST_SUCCESS = 0; // 登陆成功, 通道可用
	public final static int ST_DISCONNECT = 5; // 连接断开,不可用
	public final static int ST_NOUSER = 33; // 无此账号
	public final static int ST_PASSERR = 34; // 密码错误
	public final static int ERR_HANDLE = -1; // 错误的通道号
	static {
		try {
			if (32 == Integer.parseInt(System.getProperty("sun.arch.data.model")))
				System.loadLibrary("jniwsapi"); // 32位
			else
				System.loadLibrary("jniwsapix64"); // 64位
		} catch (UnsatisfiedLinkError e) {
			System.out.println("Cannot load library jniwsapi");
		}
	}

	/*
	 * 创建接口
	 * 
	 * @param rdburl 实时库url地址，格式为"ws://127.0.0.1:921"或者 "wss://kipway.net"
	 * 
	 * @param username 登录账号名
	 * 
	 * @param passwd 密码
	 * 
	 * @param logurl客户端日志输出服务器，格式"udp://127.0.0.1:999/rdbwsapi",如果不使用。可以
	 * 为""或者null
	 * 
	 * @return 成功返回一个大于等于0整数, -1表示失败。
	 * 
	 * @remark
	 * 返回的句柄类似与linux的fd句柄，一个进程最多同时创建64个连接，和java的文件io流对象一样，不使
	 * 用后必须显示调用Close释放句柄和内存资源。
	 */
	public static native int Create(String rdburl, String username, String passwd, String logurl);

	/*
	 * 释放由Create创建的句柄和相关的线程内存资源。
	 * 
	 * @param h 由Create创建的句柄。
	 * 
	 * @remark
	 * 句柄类似与linux的fd句柄，一个进程最多同时创建64个连接，和java的文件io流对象一样，不使用后必须显示
	 * 调用Close释放句柄和内存资源。
	 * 
	 * @return 返回0表示成功，-1表示 h参数无效
	 */
	public static native int Close(int h);

	/*
	 * 发送JSON请求消息
	 * 
	 * @param h 由Create创建的句柄。
	 * 
	 * @param req
	 * utf8的byte数组，一般由String.getBytes("UTF-8")产生。按照《rdb_data_exchange_protocol.pdf》
	 * 组织请求消息。 如果不关心seqno，可以不填写seqno字段，Call方法会自动补上。
	 * 
	 * @return 返回一个byte[]数组，是一个utf8编码的JSON字符串, 如果发生错误，status字段为非0，message字
	 * 段为错误信息。
	 */
	public static native byte[] Call(int h, byte[] req);

	/*
	 * 获取下一个seqno，用于客户端关心seqno的值时使用。
	 * 
	 * @param h 由Create创建的句柄。
	 * 
	 * @return 返回>0为下一个seqno， -1表示 h参数无效
	 */
	public static native int getNextSeqno(int h); // 获取下一个seqno; >0 成功，-1表示h通道错误

	/*
	 * 获取当前连接状态
	 * 
	 * @param h 由Create创建的句柄。
	 * 
	 * @return -1:表示h参数无效；ST_SUCCESS表示连接成功，其他ST_XXXXX为对应的错误码。
	 * 
	 * @remark 一般正常情况，Create返回后就连接成功了，除非账号，密码，服务器url错误。
	 */
	public static native int getStatus(int h);
}
