﻿/*!
\file haclient.cpp

演示双机同时连接到到热备双机自动切换场景，使用rdbapi的异步连接，主备双链接，自动重连。

本demo代码需要支持C++11规范的编译器编译:
  windows: VC140(VC2015)及以上，mt编译后目标系统无VC运行库最低可支持win7。
  linux-x64  : G++ 4.8.5及以上，参见脚本文件gccmake.sh，在wsl子系统下g++ v9.4编译通过，最低可以在centos7下G++ 4.8.5编译。
  linux-aarch64: G++ 5.4及以上版本，使用arrch64版的librdbapix64.so

bin目录为编译后的执行文件, 含windows和Linux平台，运行时动态库和执行程序放在同一目录即可。

\date 
	2024-9-25 初版
*/

//本源代码文件字符集编码为unicode utf-8带签名，代码页65001；
//源代码中字符串的编码，G++默认采用源代码文件字符集编码，VC使用本地编码或指定编码，因此VC需要指定编码，否则会按照本地编码输出可能会输出乱码。
#if defined (_MSC_VER) && _MSC_VER >= 1600  // VS2010=1600;VS2015=1900;参见 https://learn.microsoft.com/en-us/cpp/overview/compiler-versions?view=msvc-170
#pragma execution_character_set("utf-8") //设置源代码中字符串编码utf8, 或者在编译命令行设置添加 /utf-8
#endif

#ifdef _WIN32
#ifndef _CRT_SECURE_NO_WARNINGS
#define _CRT_SECURE_NO_WARNINGS 1
#endif
#pragma warning (disable : 4995)
#pragma warning (disable : 4996)
#pragma warning (disable : 4200)
#ifndef _WIN32_WINNT
#define _WIN32_WINNT 0x0601	//0x600 = vista/2008; 0x601=win7 ;  0x602=windows 8 ;  0x603=windows 8.1
#endif
#include <process.h>
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#else
#include <termios.h>
#include <unistd.h>
#include <signal.h>
#endif

#include <thread>
#include <string>
#include <vector>
#include "rdbc_tips.h"
#include "./rdbapi/rdbapi.h"
#ifdef _WIN32
#ifdef _WIN64
#pragma comment(lib, "../rdbapi/win64/rdbapix64.lib")
#else
#pragma comment(lib, "../rdbapi/win32/rdbapi.lib")
#endif
#endif

/**
 * @brief 高可用客户端
 */
class HaClient
{
public:
	HaClient() {
		_plog = new rdbc::screenLoger(); //这里创建一个屏幕输出的日志对象。
		_plog->setlevel(rdbc::logger::logv::dbg);
		_plog->open("stdout");
	}
	virtual ~HaClient() {
		Close();
		if (_plog) {
			delete _plog;
			_plog = nullptr;
		}
	}
protected:
	int _dbno{ 0 }; // 0:主库；1:从库
	int _dbh{ -1 }; //实时库句柄, 代表了一个消息循环线程和一些使用的内存和网络连接资源。-1表示无效句柄。
	int _status{ -1 }; //实时库登录状态，-1未登录; 0:已登录
	int _subscribed{ 0 };//已订阅标识 0:未订阅; 1:已订阅;
	rdbc::logger* _plog{ nullptr }; //日志输出接口
	int _nextSeqNo{ 1 };

	//{{保存登录信息，除了显示之外没什么用, rdbapi本身自带断线重连
	std::string _protocol; //协议 "ws" or "wss"
	std::string _wsrul; //实时库连接url
	std::string _username;//用户账号
	std::string _passwd;//登录密码

	unsigned short  _slaveport{ 921 };
	std::string _slaveip;
	std::string _slaveuser;
	std::string _slavepass;
	//}}

	int NextSeqNo()
	{
		if (_nextSeqNo < INT32_MAX)
			return _nextSeqNo++;
		_nextSeqNo = 1;
		return _nextSeqNo++;
	}

	/**
	 * @brief 当前连接的库信息
	 * @param url 输出的连接url
	 * @return 返回0：主库； 1：从库
	 */
	const char* curdburl(std::string& url)
	{
		if (_dbh >= 0) {
			_dbno = rdb_curdb(_dbh); //连接的库
		}
		if (_dbno) { //从库
			url.append(_protocol.c_str()).append("://").append(_slaveip).append(":").append(std::to_string(_slaveport));
		}
		else {
			url = _wsrul;
		}
		return url.c_str();
	}
private:
	/**
	 * @brief 事件通知回调
	 * @param nmsgcode 消息码
	 * @param smsg 消息内容
	 * @param param 回调参数，这里压入this指针.
	 * @remark 注意这个函数在rdbapi的消息循环线程里被调用，需要做好多线程同步和临界区保护。这里只是输出到日志，没有什么需要同步和保护的。
	 * 不要在这个回调函数例调用其他的需要和实时库交互的rdb_xxx函数(简单的说就是带连接句柄h参数的函数)，可以调用不带句柄参数的工具函数。
	 * 如果要在连接成功后订阅快照，可以设置一个标识，在应用程序主消息循环里订阅。其他回调函数也是一样的不能再调用交互函数。
	 */
	static void cd_onmessage(int nmsgcode, const char* smsg, void* param)
	{
		HaClient* pcls = (HaClient*)param;
		if (MSGCODE_CONNECT_SUCCESS == nmsgcode) {
			std::string cururl;
			pcls->_dbno = rdb_curdb(pcls->_dbh); //连接的库,rdb_curdb不和实时库交互。
			pcls->_subscribed = 0; //设置未订阅标识，登录成功后需要订阅，由于不能在回调函数里调用和实时库交互的函数，因此这里只设置标识，在主循环里订阅。
			pcls->_status = 0;//设置登录成功
			pcls->_plog->out(rdbc::logger::logv::inf, "connect %s success", pcls->curdburl(cururl));			
		}
		else if (MSGCODE_CONNECT_FAILED == nmsgcode) {
			std::string cururl;
			pcls->_plog->out(rdbc::logger::logv::err, "connect %s failed, %s", pcls->curdburl(cururl), smsg ? smsg : "");
		}
		else if (MSGCODE_DISCONNECTED == nmsgcode) {
			std::string cururl;
			pcls->_status = -1;//设置断开
			pcls->_plog->out(rdbc::logger::logv::wrn, "rdb %s is disconnected", pcls->curdburl(cururl));
		}
	}

	/**
	 * @brief 接收SOE主动推送消息的回调函数.
	 * @param psoes SOE数组
	 * @param nitems SOE个数
	 * @param param 回调参数，这里压入this指针.
	 * @remark 注意这个函数在rdbapi的消息循环线程里被调用，需要做好多线程同步和临界区保护，不要在这个回调函数例调用其他的需要和实时库交互的接口函数。
	 */
	static void cb_OnSscPutSoes(rec_soe psoes[], int nitems, void* param)
	{
		HaClient* pcls = (HaClient*)param;
		//处理实时库推送的SOE回调, 这里只是输出到日志，如果要存入你的集合做其他处理，需要加锁。
		pcls->_plog->out(rdbc::logger::logv::inf, "recv %d SOE", nitems);
		char stime[48] = { 0 };
		char utf8sourse[160], utf8des[320]; //控制台是utf8编码，rec_soe里是gbk编码，这里需要转码后输出到控制台日志。
		utf8sourse[0] = '\0';
		utf8des[0] = '\0';
		const char* ssrc = nullptr, * sdes = nullptr;
		for (auto i = 0; i < nitems; i++) {
			rdb_jtime2string(psoes[i].time * 100ll, stime, sizeof(stime), 1);
			ssrc = psoes[i].source;
			sdes = psoes[i].sdes;
			if (!rdbc::strisascii(ssrc)) {
				if (rdb_gbk2utf8(ssrc, strlen(ssrc), utf8sourse, sizeof(utf8sourse)) >= 0)
					ssrc = utf8sourse;
			}
			if (!rdbc::strisascii(sdes)) {
				if (rdb_gbk2utf8(sdes, strlen(sdes), utf8des, sizeof(utf8des)) >= 0)
					sdes = utf8des;
			}
			pcls->_plog->out(rdbc::logger::logv::inf, "\ntime=%jd, %s; source=%s; sdes=%s;",
				(int64_t)(psoes[i].time * 100ll), //实时库数字时标,1970-1-1开始的UTC时标，单位100毫秒数, *100转为毫秒
				stime,
				ssrc,//事件源
				sdes//事件描述
			);
		}
	}

	/**
	 * @brief 处理服务器推送的订阅标签快照值数据回调函数
	 * @param pvals 记录集
	 * @param sizeVals 记录数
	 * @param param 设置回调函数时的参数。
	 * @remark 不要在这个回调函数例调用其他的需要和实时库交互的接口函数。
	 */
	static void cb_OnPushValSnaps(const rec_tagval pvals[], int sizeVals, void* param)
	{
		size_t zlen;
		const char* utf8name;
		char utf8tmp[200], val[80], stime[40];
		HaClient* pcls = (HaClient*)param;
		pcls->_plog->out(rdbc::logger::logv::dbg, "cb_OnPushValSnaps %d records.", sizeVals);//这里只是输出到日志。
		for (auto i = 0; i < sizeVals; i++) {
			utf8name = pvals[i].sname;
			zlen = strlen(utf8name);
			if (!rdbc::strisascii(utf8name, zlen)) { //不是ascii码，因为控制到输出是utf8字符集设置，需要转换为utf8编码。
				if (rdb_gbk2utf8(utf8name, zlen, utf8tmp, sizeof(utf8tmp)) > 0) {
					utf8name = (const char*)utf8tmp;
				}
			}
			switch (pvals[i].val.cvt) {
			case DT_DIGITAL:
			case DT_INT32:
				sprintf(val, "%d", pvals[i].val.i32);
				break;
			case DT_FLOAT32:
				sprintf(val, "%f", pvals[i].val.f32);
				break;
			case DT_INT64:
				sprintf(val, "%jd", (int64_t)pvals[i].val.i64);
				break;
			case DT_FLOAT64:
				sprintf(val, "%f", pvals[i].val.f64);
				break;
			default:
				val[0] = '\0';
				break;
			}
			rdb_jtime2string(pvals[i].val.time * 100ll, stime, sizeof(stime), 1);
			pcls->_plog->append(rdbc::logger::logv::dbg, "    %s, time=%s, QA=%d, datatype=%d(%s), val=%s\n",
				utf8name, stime, pvals[i].val.cqa, pvals[i].val.cvt, rdbc::typeStr(pvals[i].val.cvt), val);
		}
	}

	/**
	 * @brief 处理服务器推送的订阅标签快照对象数据回调函数
	 * @param pobjs 记录集
	 * @param sizeObjs 记录数
	 * @param param 设置回调函数时的参数。
	 * @remark 不要在这个回调函数例调用其他的需要和实时库交互的接口函数。
	 */
	static void cb_OnPushObjSnaps(const rec_tagobj pobjs[], int sizeObjs, void* param)
	{
		size_t zlen;
		const char* utf8name;
		char utf8tmp[200], val[80], stime[40];
		HaClient* pcls = (HaClient*)param;
		pcls->_plog->out(rdbc::logger::logv::dbg, "cb_OnPushObjSnaps %d records.", sizeObjs);//这里只是输出到日志。
		for (auto i = 0; i < sizeObjs; i++) {
			utf8name = pobjs[i].sname;
			zlen = strlen(utf8name);
			if (!rdbc::strisascii(utf8name, zlen)) { //不是ascii码，因为控制到输出是utf8字符集设置，需要转换为utf8编码。
				if (rdb_gbk2utf8(utf8name, zlen, utf8tmp, sizeof(utf8tmp)) > 0) {
					utf8name = (const char*)utf8tmp;
				}
			}
			rdb_jtime2string(pobjs[i].var.time * 100ll, stime, sizeof(stime), 1);
			if (pobjs[i].var.cvt == DT_STRING) {
				pcls->_plog->append(rdbc::logger::logv::dbg, "    %s, time=%s, QA=%d, strlen=%u, string=%.*s\n",
					utf8name, stime, pobjs[i].var.cqa, pobjs[i].var.uslen, pobjs[i].var.uslen < 80 ? pobjs[i].var.uslen : 80, pobjs[i].var.sdata);
			}
			else {
				rdbc::hex2str(pobjs[i].var.sdata, pobjs[i].var.uslen, val, sizeof(val)); //显示前不分内容
				pcls->_plog->append(rdbc::logger::logv::dbg, "    %s, time=%s, QA=%d, objectsize=%u, object=%s\n",
					utf8name, stime, pobjs[i].var.cqa, pobjs[i].var.uslen, val);
			}
		}
	}

	/**
	 * @brief 处理服务器推送的JSON消息(包括SOE和订阅数据推送)回调函数,适合高级用户自己解析订阅的SOE和订阅的快照数据JSON报文
	 * @param jstr 接收到服务器推送的JSON消息,utf8编码
	 * @param sizejstr 消息长度.
	 * @param param 设置回调时压入的参数.
	 * @remark 高级用户想自己解析服务器推送的快照和SOE消息才需要安装这个回调函数。安装此回调后，上面三个回调cb_OnSscPutSoes，cb_OnPushValSnaps，cb_OnPushObjSnaps
	 * 就不需要安装了，如果同时安装会同时生效被同时回调。不要在这个回调函数例调用其他的需要和实时库交互的接口函数。
	 */
	static void cb_OnPushJsonMsgs(const char* jstr, int sizejstr, void* param)
	{
		HaClient* pcls = (HaClient*)param;
		pcls->_plog->out(rdbc::logger::logv::dbg, "cb_OnPushJsonMsgs:\n%.*s", sizejstr < 4000 ? sizejstr : 4000, jstr); 
		//输出前4000字符到日志演示以下，实际应该解析JSON消息，根据协议处理。
	}
public:
	/**
	 * @brief 创建实时库句柄，设置参数，开启异步连接
	 * @param surl 主库实时库连接ulr, "wss://kipway.net" or "ws://192.168.1.214:921"
	 * @param suser 主库用户名 "opt1"
	 * @param spasswd 主库密码 "opt1"
	 * @param slaveip 从库ip(不能是域名, 在今后的版本可能支持域名),只能是点分格式例如"192.168.1.59"
	 * @param slaveport 从端口，比如 921
	 * @param slaveuser 从库账号
	 * @param slavepass 从库密码
	 * @return 0:OK; -1:error;
	 * @remark 不管是单服务器还是双服务器连接，断开后都会自动重连的，连接成功失败使用回调函数cd_onmessage通知。
	 */
	int Open(const char* surl, const char* suser, const char* spasswd,
		const char* slaveip, unsigned short slaveport, const char* slaveuser, const char* slavepass)
	{
		if (-1 != _dbh) {
			return 0;
		}
		if (!surl || !*surl || !suser || !*suser || !spasswd || !*spasswd) {
			_plog->out(rdbc::logger::logv::err, "error master args");
			return -1; //参数错误
		}
		if (!slaveport || !slaveip || !*slaveip || !slaveuser || !*slaveuser || !slavepass || !*slavepass) {
			_plog->out(rdbc::logger::logv::err, "error slave args");
			return -1; //参数错误
		}

		_dbh = rdb_create();
		if (_dbh < 0) {
			_plog->out(rdbc::logger::logv::err, "rdb_create failed.");
			return -1;
		}
		rdb_setmessagenotify(_dbh, cd_onmessage, this);//设置消息回调，用于异步连接。
		rdb_soesubscription_setfun(_dbh, cb_OnSscPutSoes, this); //设置SOE处理回调,订阅SOE才需要		
		rdb_subscription_setfun(_dbh, cb_OnPushValSnaps, this, cb_OnPushObjSnaps, this); //设置订阅标签快照推送接收处理
		rdb_pushjsonmessage_setfun(_dbh, cb_OnPushJsonMsgs, this);//高级用户，可选设置，自己处理服务器推送消息。
		//实际应用中，使用rdb_pushjsonmessage_setfun后，rdb_soesubscription_setfun和rdb_subscription_setfun就不需要再设置了。
		//这里例子两个都设置，都会被回调。
		
		_protocol.assign(surl, 3);
		if (stricmp(_protocol.c_str(), "wss")) {
			_protocol = "ws";
		}
		rdb_setslavedb(_dbh, slaveip, slaveport, slaveuser, slavepass);//设置从库连接参数
		_slaveip = slaveip;
		_slaveport = slaveport;

		int ncode = rdb_connectex(_dbh, surl, suser, spasswd, 0);//最后再异步连接
		if (SE_OK != ncode) {
			_plog->out(rdbc::logger::logv::err, "connect %s failed. errcode = %d", surl, ncode);
			return -1;
		}
		
		_wsrul = surl; //保存供以后显示用。
		_username = suser;
		_passwd = spasswd;
		return 0;
	}

	/**
	 * @brief 关闭,应用退出时调用,断开连接，终止运行时线程和释放其他资源。
	 * @return 0
	 */
	int Close()
	{
		if (-1 == _dbh)
			return 0;
		rdb_disconnect(_dbh); //断开连接
		rdb_destory(_dbh);//销毁句柄
		_dbh = -1;//设置句柄无效，避免重复销毁
		return 0;
	}

	/**
	 * @brief 运行时循环,这里只做一件事，检查_subscribed标识并重新订阅。
	 */
	void runtime()
	{
		if (0 == _subscribed && 0 == _status) { //已登录且未订阅
			int ndbver = rdb_curdb_inver(_dbh);
			std::string dburl;
			curdburl(dburl);
			if (ndbver < 5119) {
				_plog->out(rdbc::logger::logv::wrn, "rdb %s inver=%d, not support subscribe", dburl.c_str(), ndbver);
			}
			else {
				std::vector<const char*> tags{ "d0.f0*","d1.str01.pv","d1.obj01.pv" };
				int nret = rdb_sscsnaps(_dbh, 1, 1, tags.data(), (int)tags.size(), [](const char* tagname, int errcode, void* param) {
					HaClient* pcls = (HaClient*)param;
					pcls->_plog->out(rdbc::logger::logv::err, "subscribe %s error %d", tagname, errcode);
					}, this);
				if (nret != SE_OK) {
					_plog->out(rdbc::logger::logv::err, "rdb %s inver=%d, rdb_sscsnaps failed, errcode %d", dburl.c_str(), ndbver, nret);
				}
				else {
					_plog->out(rdbc::logger::logv::inf, "rdb %s inver=%d, rdb_sscsnaps success.", dburl.c_str(), ndbver);
				}
			}
			_subscribed = 1;//置已订阅标识
		}
	}
};

int g_stop = 0; //停止标识

#ifdef _WIN32
BOOL WINAPI exit_HandlerRoutine(DWORD dwCtrlType) 
{
	printf("dwCtrlType %u\n", dwCtrlType);
	switch (dwCtrlType) {
	case CTRL_C_EVENT:
	case CTRL_BREAK_EVENT:
	case CTRL_CLOSE_EVENT:
	case CTRL_SHUTDOWN_EVENT:
		g_stop = 1;
		break;
	default:
		return FALSE;
	}
	return TRUE;
}
#else
void exit_handler(int sigval)
{
	if (sigval) {
		signal(sigval, SIG_IGN);
	}
	g_stop = 1;
}
#endif

//usage:
// url user pass slaveip slaveport [slaveuser] [slavepass]

const char* shelp = "usage:\n"
"  haclient url user pass slaveip slaveport [slaveuser] [slavepass]\n"
"demo:\n"
"  haclient ws://192.168.1.214:921 admin admin 192.168.1.59 921\n"
"  haclient ws://192.168.1.214:921 admin admin 192.168.1.59 921 admin admin\n"
"\n"
;

void usage()
{
	printf("%s", shelp);
}

int main(int argc, const char* argv[])
{
#ifdef _WIN32
	SetConsoleOutputCP(65001); //设置控制台输出字符集utf8编码，和linux默认字符集一致。
#endif
	if (argc < 6 || argc > 8) {
		usage();
		return 0;
	}

	//安装控制台事件处理函数
#ifdef _WIN32	
	SetConsoleCtrlHandler(exit_HandlerRoutine, TRUE);
#else	
	signal(SIGPIPE, SIG_IGN);
	signal(SIGTERM, exit_handler);
	signal(SIGINT, exit_handler);
#endif

	//解析命令行参数
	const char* slaveuser = argv[2];//账号默认和主库相同
	const char* slavepass = argv[3];//密码默认和主库相同
	if (argc > 6 && argv[6])
		slaveuser = argv[6];
	if (argc > 7 && argv[7])
		slavepass = argv[7];

	HaClient hac;
	if (hac.Open(argv[1], argv[2], argv[3], argv[4], (uint16_t)atoi(argv[5]), slaveuser, slavepass) < 0) {
		return -1;
	}
	printf("Ctrl + C to exit!\n"); //不要在VC的调试环境下按Ctrl+C或被调试器拦截，使用Ctrl+break

	while (!g_stop) { //主循环
		hac.runtime();// 检查并处理订阅标识。
		std::this_thread::sleep_for(std::chrono::milliseconds(100));//主线程只是检查重新订阅标识，这里延迟只是为了降低cpu占用。
		//这里得延迟并不会影响rdbapi后台线程得数据接收处理。这个例子在进程里看到是两个线程，这个主线程和rdbapi网络io和数据处理线程。
	}
	hac.Close();
	printf("haclient exit success!\n");
	return 0;
}