﻿/*!
* \file rdbc_tips.hpp
* rdb C++ demo 一些工具类，比如日志和时间
*/

#pragma once
#include <cstdint>
#include <cstdarg>
#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <time.h>
#include <string>

#ifdef _WIN32
#define strcasecmp stricmp
#else
#include <fcntl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <errno.h>
#define _tzset tzset
#define  stricmp strcasecmp
#ifndef SOCKET
#define SOCKET int
#endif

#ifndef TIMEVAL
#define TIMEVAL struct timeval
#endif
#endif

#define ECTIME_LOCALSTR 0 // "2023/2/20 12:10:32.123"    lcoal datetime
#define ECTIME_ISOSTR   1 // "2023-2-20T12:10:32.123+08:00"  ISO 8601
#define ECTIME_STAMP    2 // 1676866232123     UTC milliseconds since 1970-1-1

#include "./rdbapi/rdbapi.h"
namespace rdbc
{
	/**
	 * @brief 转换位16进制书写格式
	 * @param psrc 输入
	 * @param sizesrc 输入字节数
	 * @param sout 输出，最后一定会输出0结束
	 * @param outsize 输出空间大小
	 */
	inline void hex2str(const void* psrc, size_t sizesrc, char* sout, size_t outsize)
	{
		unsigned char uc;
		size_t i;
		const unsigned char* pu = (const unsigned char*)psrc;
		for (i = 0; i < sizesrc && 2 * i + 2 < outsize; i++) {
			uc = pu[i] >> 4;
			sout[i * 2] = (uc >= 0x0A) ? 'A' + (uc - 0x0A) : '0' + uc;
			uc = pu[i] & 0x0F;
			sout[i * 2 + 1] = (uc >= 0x0A) ? 'A' + (uc - 0x0A) : '0' + uc;
		}
		sout[2 * i] = 0;
	}

	/**
	* @brief 判断字符串是否全是ASCII码，如果是，那一定也是UTF8编码
	* @param s 输入的字符串
	* @param size 字符串长度，如果是默认值0，则s会按照0结束处理使用strlen计算
	* @return true 全是ASCII码； false：不全是
	*/
	inline bool strisascii(const char* s, size_t size = 0)
	{
		if (!s)
			return true;
		const char* pend = s + (size ? size : strlen(s));
		while (s < pend) {
			if (*s & 0x80)
				return false;
			++s;
		}
		return true;
	}

	inline int64_t mstime() // Return milliseconds since 1970-1-1
	{
#ifdef _WIN32
		FILETIME ft;
		GetSystemTimeAsFileTime(&ft);
		ULARGE_INTEGER ul;
		ul.LowPart = ft.dwLowDateTime;
		ul.HighPart = ft.dwHighDateTime;
		return (int64_t)((ul.QuadPart / 10000000LL) - 11644473600LL) * 1000 + (ul.QuadPart % 10000000LL) / 10000;
#else
		struct timeval tv;
		gettimeofday(&tv, nullptr);
		return (int64_t)(tv.tv_sec * 1000LL + tv.tv_usec / 1000);
#endif
	}

	/**
	 * @brief rec_val值转换位字符串用于显示
	 * @param r 记录引用
	 * @param sout 输出的字符串引用
	 */
	inline void recValtoString(struct rec_val& r, std::string& sout)
	{
		sout.clear();
		switch (r.cvt)
		{
		case DT_DIGITAL:
		case DT_INT32:
			sout = std::move(std::to_string(r.i32));
			break;
		case DT_FLOAT32:
			sout = std::move(std::to_string(r.f32));
			break;
		case DT_INT64:
			sout = std::move(std::to_string(r.i64));
			break;
		case DT_FLOAT64:
			sout = std::move(std::to_string(r.f64));
			break;
		}
	}

	/**
	 * @brief 数据类型转换为可视字符串
	 * @param datatype
	 * @return 返回类型对应的字符串
	 */
	inline const char* typeStr(int datatype)
	{
		const char* s = "void";
		switch (datatype)
		{
		case DT_DIGITAL:
			s = "digital";
			break;
		case DT_INT32:
			s = "int32";
			break;
		case DT_FLOAT32:
			s = "float";
			break;
		case DT_INT64:
			s = "int64";
			break;
		case DT_FLOAT64:
			s = "double";
			break;
		case DT_STRING:
			s = "string";
			break;
		case DT_OBJECT:
			s = "object";
			break;
		}
		return s;
	}

	/**
	 * @brief 日志输出基类
	 */
	class logger
	{
	public:		
		enum class logv {
			err = 100,
			wrn = 200,
			inf = 300,
			dbg = 400,
			all = 500
		};		
	protected:
		logv _level = logv::inf;

	public:
		virtual ~logger() {
		}
		inline logv getlevel() {
			return _level;
		}
		inline void setlevel(logv level) {
			_level = level;
		}
		virtual int open(const char* args) = 0; // return -1:error; 0:success
#ifdef _WIN32
		virtual int out(logv level, const char* format, ...) = 0; //return >= 0 out string length; -1:error;
#else
		virtual int out(logv level, const char* format, ...) __attribute__((format(printf, 3, 4))) = 0; //return >= 0 out string length; -1:error;
#endif
#ifdef _WIN32
		virtual int append(logv level, const char* format, ...) = 0; //return >= 0 out string length; -1:error;
#else
		virtual int append(logv level, const char* format, ...) __attribute__((format(printf, 3, 4))) = 0; //return >= 0 out string length; -1:error;
#endif
		static const char* level_str(logv n, char* sout = nullptr, size_t soutsize = 0)
		{
			const char* sret = "ukn";
			switch (n) {
			case logv::err :
				sret = "err";
				break;
			case logv::wrn:
				sret = "wrn";
				break;
			case logv::inf:
				sret = "inf";
				break;
			case logv::dbg:
				sret = "dbg";
				break;
			case logv::all:
				sret = "all";
				break;
			}
			return sret;
		}
	};

	/**
	 * @brief 这里实现一个屏幕输出日志，你可实现其他的输出到文件或者网络的日志类。
	 */
	class screenLoger : public logger
	{
	private:
		const char* colorstr(logv lev) {
			const char* s = "\033[m";
			switch (lev) {
			case logv::err:
				s = "\033[0;31m"; //red
				break;
			case logv::wrn:
				s = "\033[0;33m"; //yellow
				break;
			case logv::inf:
				s = "\033[0;32m"; //green
				break;
			case logv::dbg:
				break;
			case logv::all:
				break;
			}
			return s;
		}
	public:
		enum out_type {
			out_null = 0, // out to null
			out_std = 1 // out to stdout
		};
		/**
		 * @brief 打开
		 * @param args  "stdout" , or "nul"
		 * @return
		 */
		int open(const char* args)  override
		{
			if (args && *args) {
				if (!stricmp("stdout", args))
					_outtype = out_std;
				else
					_outtype = out_null;
			}
			else
				_outtype = out_null;
			return 0;
		}
#ifdef _WIN32
		virtual int out(logv level, const char* format, ...) //return >= 0 out string length; -1:error;
#else
		virtual int out(logv level, const char* format, ...) __attribute__((format(printf, 3, 4))) //return >= 0 out string length; -1:error;
#endif
		{
			if (_outtype == out_null || _level < level)
				return 0;
			int64_t curms = mstime();
			time_t gmt = (time_t)(curms / 1000);

			struct tm tml;
#ifdef _WIN32
			if (localtime_s(&tml, &gmt))
				return false;
#else
			if (!localtime_r(&gmt, &tml))
				return false;
#endif
			char slev[16] = { 0 };
			printf("[%d-%d-%d %02d:%02d:%02d.%03d] %s[%s] "
				, tml.tm_year + 1900, tml.tm_mon + 1, tml.tm_mday, tml.tm_hour, tml.tm_min, tml.tm_sec, (int)(curms % 1000)
				, colorstr(level), level_str(level, slev, sizeof(slev)));
			va_list arg_ptr;
			va_start(arg_ptr, format);
			int n = vprintf(format, arg_ptr);
			va_end(arg_ptr);
			printf("\033[m\n");
			return n;
		}
#ifdef _WIN32
		virtual int append(logv level, const char* format, ...) //return >= 0 out string length; -1:error;
#else
		virtual int append(logv level, const char* format, ...) __attribute__((format(printf, 3, 4))) //return >= 0 out string length; -1:error;
#endif
		{
			printf("%s", colorstr(level));
			va_list arg_ptr;
			va_start(arg_ptr, format);
			int n = vprintf(format, arg_ptr);
			va_end(arg_ptr);
			printf("\033[m");
			return n;
		}
	protected:
		out_type _outtype = out_null;
	};
}// namespace  rdbc


