﻿/*!
\file rdbcppdemo.cpp
rdbapi本身是一个纯C的标准动态库，这里使用C++来演示应用层如何使用。

本demo代码需要支持C++11规范的编译器编译:
  windows: VC140(VC2015)及以上，mt编译后目标系统无VC运行库最低可支持win7。
  linux-x64  : G++ 4.8.5及以上，参见脚本文件gccmake.sh，在wsl子系统下g++ v9.4编译通过，最低可以在centos7下G++ 4.8.5编译。
  linux-aarch64: G++ 5.4及以上版本，使用arrch64版的librdbapix64.so

bin目录为编译后的执行文件, 含windows和Linux平台，运行时动态库和执行程序放在同一目录即可。

rdbapi的应用流程，初始化参数 ->异步连接->事件通知处理,接口调用(循环)->断开关闭；
另外提供了高级用法，根据《rdb_data_exchange_protocol.pdf》直接和实时库交互，建议C++用户直接使用 cDemo::Call()演示的方法直接和实时库交互。
对于更高级的C++用户，建议抛弃rdbapi，直接使用websocket和《rdb_data_exchange_protocol.pdf》协议和实时库交互，使用异步模式更高效和自由。

\date 
	2024-9-22 增加写快照例子和订阅例子
	2024-8-23 第一次发布
*/

//本源代码文件字符集编码为unicode utf-8带签名，代码页65001；
//源代码中字符串的编码，G++默认采用源代码文件字符集编码，VC使用本地编码或指定编码，因此VC需要指定编码，否则会按照本地编码输出可能会输出乱码。
#if defined (_MSC_VER) && _MSC_VER >= 1600  // VS2010=1600;VS2015=1900;参见 https://learn.microsoft.com/en-us/cpp/overview/compiler-versions?view=msvc-170
#pragma execution_character_set("utf-8") //设置源代码中字符串编码utf8, 或者在编译命令行设置添加 /utf-8
#endif

#ifdef _WIN32
#ifndef _CRT_SECURE_NO_WARNINGS
#define _CRT_SECURE_NO_WARNINGS 1
#endif
#pragma warning (disable : 4995)
#pragma warning (disable : 4996)
#pragma warning (disable : 4200)
#ifndef _WIN32_WINNT
#define _WIN32_WINNT 0x0601	//0x600 = vista/2008; 0x601=win7 ;  0x602=windows 8 ;  0x603=windows 8.1
#endif
#include <process.h>
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#else
#include <termios.h>
#include <unistd.h>
#endif

#include <string>
#include <vector>
#include "rdbc_tips.h"
#ifdef _WIN32
#ifdef _WIN64
#pragma comment(lib, "./rdbapi/win64/rdbapix64.lib")
#else
#pragma comment(lib, "./rdbapi/win32/rdbapi.lib")
#endif
#endif

/**
 * @brief 应用例子
 */
class cDemo
{
public:
	cDemo() {
		_plog = new rdbc::screenLoger(); //这里创建一个屏幕输出的日志对象。
		_plog->setlevel(rdbc::logger::logv::dbg);
		_plog->open("stdout");
	}
	virtual ~cDemo() {
		Close();
		if (_plog) {
			delete _plog;
			_plog = nullptr;
		}
	}
	inline int status() {
		return _status;
	}
protected:
	int _dbh{ -1 }; //实时库句柄, 代表了一个消息循环线程和一些使用的内存和网络连接资源。-1表示无效句柄。
	int _status{ -1 }; //实时库登录状态，-1未登录; 0:已登录
	rdbc::logger* _plog{ nullptr }; //日志输出接口
	int _nextSeqNo{ 1 };

	//{{保存登录信息，除了显示之外没什么用, rdbapi本身自带断线重连
	std::string _wsrul; //实时库连接url
	std::string _username;//用户账号
	std::string _passwd;//登录密码
	//}}

	int NextSeqNo()
	{
		if (_nextSeqNo < INT32_MAX)
			return _nextSeqNo++;
		_nextSeqNo = 1;
		return _nextSeqNo++;
	}
private:
	/**
	 * @brief 事件通知回调
	 * @param nmsgcode 消息码
	 * @param smsg 消息内容
	 * @param param 回调参数，这里压入this指针.
	 * @remark 注意这个函数在rdbapi的消息循环线程里被调用，需要做好多线程同步和临界区保护。这里只是输出到日志，没有什么需要同步和保护的。
	 * 不要在这个回调函数例调用其他的需要和实时库交互的rdb_xxx函数(简单的说就是带连接句柄h参数的函数)，可以调用不带句柄参数的工具函数。
	 * 如果要在连接成功后订阅快照，可以设置一个标识，在应用程序主消息循环里订阅。其他回调函数也是一样的不能再调用交互函数。
	 */
	static void cd_onmessage(int nmsgcode, const char* smsg, void* param)
	{
		cDemo* pcls = (cDemo*)param;
		if (MSGCODE_CONNECT_SUCCESS == nmsgcode) {
			pcls->_status = 0;//设置登录成功
			pcls->_plog->out(rdbc::logger::logv::inf, "connect %s success", pcls->_wsrul.c_str());
			//如果要使用自动订阅，在这里设置需要重新订阅的标识，然后在应用程序主循环里检查这个标识，重新订阅后清除标识。
			//后面会单独写一个重连后自动重新订阅的双机热备客户端例子。
		}
		else if (MSGCODE_CONNECT_FAILED == nmsgcode) {
			pcls->_plog->out(rdbc::logger::logv::err, "connect %s failed, %s", pcls->_wsrul.c_str(), smsg ? smsg : "");
		}
		else if (MSGCODE_DISCONNECTED == nmsgcode) {
			pcls->_status = -1;//设置断开
			pcls->_plog->out(rdbc::logger::logv::wrn, "rdb %s is disconnected", pcls->_wsrul.c_str());
		}
	}

	/**
	 * @brief 接收SOE主动推送消息的回调函数.
	 * @param psoes SOE数组
	 * @param nitems SOE个数
	 * @param param 回调参数，这里压入this指针.
	 * @remark 注意这个函数在rdbapi的消息循环线程里被调用，需要做好多线程同步和临界区保护，不要在这个回调函数例调用其他的需要和实时库交互的接口函数。
	 */
	static void cb_OnSscPutSoes(rec_soe psoes[], int nitems, void* param)
	{
		cDemo* pcls = (cDemo*)param;
		//处理实时库推送的SOE回调, 这里只是输出到日志，如果要存入你的集合做其他处理，需要加锁。
		pcls->_plog->out(rdbc::logger::logv::inf, "recv %d SOE", nitems);
		char stime[48] = { 0 };
		char utf8sourse[160], utf8des[320]; //控制台是utf8编码，rec_soe里是gbk编码，这里需要转码后输出到控制台日志。
		utf8sourse[0] = '\0';
		utf8des[0] = '\0';
		const char* ssrc = nullptr, * sdes = nullptr;
		for (auto i = 0; i < nitems; i++) {
			rdb_jtime2string(psoes[i].time * 100ll, stime, sizeof(stime), 1);
			ssrc = psoes[i].source;
			sdes = psoes[i].sdes;
			if (!rdbc::strisascii(ssrc)) {
				if (rdb_gbk2utf8(ssrc, strlen(ssrc), utf8sourse, sizeof(utf8sourse)) >= 0)
					ssrc = utf8sourse;
			}
			if (!rdbc::strisascii(sdes)) {
				if (rdb_gbk2utf8(sdes, strlen(sdes), utf8des, sizeof(utf8des)) >= 0)
					sdes = utf8des;
			}
			pcls->_plog->out(rdbc::logger::logv::inf, "\ntime=%jd, %s; source=%s; sdes=%s;",
				(int64_t)(psoes[i].time * 100ll), //实时库数字时标,1970-1-1开始的UTC时标，单位100毫秒数, *100转为毫秒
				stime,
				ssrc,//事件源
				sdes//事件描述
			);
		}
	}

	/**
	 * @brief 处理服务器推送的订阅标签快照值数据回调函数
	 * @param pvals 记录集
	 * @param sizeVals 记录数
	 * @param param 设置回调函数时的参数。
	 * @remark 不要在这个回调函数例调用其他的需要和实时库交互的接口函数。
	 */
	static void cb_OnPushValSnaps(const rec_tagval pvals[], int sizeVals, void* param)
	{
		size_t zlen;
		const char* utf8name;
		char utf8tmp[200], val[80], stime[40];
		cDemo* pcls = (cDemo*)param;
		pcls->_plog->out(rdbc::logger::logv::dbg, "cb_OnPushValSnaps %d records.", sizeVals);
		for (auto i = 0; i < sizeVals; i++) {
			utf8name = pvals[i].sname;
			zlen = strlen(utf8name);
			if (!rdbc::strisascii(utf8name, zlen)) { //不是ascii码，因为控制到输出是utf8字符集设置，需要转换为utf8编码。
				if (rdb_gbk2utf8(utf8name, zlen, utf8tmp, sizeof(utf8tmp)) > 0) {
					utf8name = (const char*)utf8tmp;
				}
			}
			switch (pvals[i].val.cvt) {
			case DT_DIGITAL:
			case DT_INT32:
				sprintf(val, "%d", pvals[i].val.i32);
				break;
			case DT_FLOAT32:
				sprintf(val, "%f", pvals[i].val.f32);
				break;
			case DT_INT64:
				sprintf(val, "%jd", (int64_t)pvals[i].val.i64);
				break;
			case DT_FLOAT64:
				sprintf(val, "%f", pvals[i].val.f64);
				break;
			default:
				val[0] = '\0';
				break;
			}
			rdb_jtime2string(pvals[i].val.time * 100ll, stime, sizeof(stime), 1);
			pcls->_plog->append(rdbc::logger::logv::dbg, "    %s, time=%s, QA=%d, datatype=%d(%s), val=%s\n",
				utf8name, stime, pvals[i].val.cqa, pvals[i].val.cvt, rdbc::typeStr(pvals[i].val.cvt), val);
		}
	}

	/**
	 * @brief 处理服务器推送的订阅标签快照对象数据回调函数
	 * @param pobjs 记录集
	 * @param sizeObjs 记录数
	 * @param param 设置回调函数时的参数。
	 * @remark 不要在这个回调函数例调用其他的需要和实时库交互的接口函数。 
	 */
	static void cb_OnPushObjSnaps(const rec_tagobj pobjs[], int sizeObjs, void* param)
	{
		size_t zlen;
		const char* utf8name;
		char utf8tmp[200], val[80], stime[40];
		cDemo* pcls = (cDemo*)param;
		pcls->_plog->out(rdbc::logger::logv::dbg, "cb_OnPushObjSnaps %d records.", sizeObjs);
		for (auto i = 0; i < sizeObjs; i++) {
			utf8name = pobjs[i].sname;
			zlen = strlen(utf8name);
			if (!rdbc::strisascii(utf8name, zlen)) { //不是ascii码，因为控制到输出是utf8字符集设置，需要转换为utf8编码。
				if (rdb_gbk2utf8(utf8name, zlen, utf8tmp, sizeof(utf8tmp)) > 0) {
					utf8name = (const char*)utf8tmp;
				}
			}			
			rdb_jtime2string(pobjs[i].var.time * 100ll, stime, sizeof(stime), 1);
			if (pobjs[i].var.cvt == DT_STRING) {
				pcls->_plog->append(rdbc::logger::logv::dbg, "    %s, time=%s, QA=%d, strlen=%u, string=%.*s\n",
					utf8name, stime, pobjs[i].var.cqa, pobjs[i].var.uslen, pobjs[i].var.uslen < 80 ? pobjs[i].var.uslen : 80, pobjs[i].var.sdata);
			}
			else {
				rdbc::hex2str(pobjs[i].var.sdata, pobjs[i].var.uslen, val, sizeof(val)); //显示前不分内容
				pcls->_plog->append(rdbc::logger::logv::dbg, "    %s, time=%s, QA=%d, objectsize=%u, object=%s\n",
					utf8name, stime, pobjs[i].var.cqa, pobjs[i].var.uslen, val);
			}
		}
	}

	/**
	 * @brief 处理服务器推送的JSON消息(包括SOE和订阅数据推送)回调函数,适合高级用户自己解析订阅的SOE和订阅的快照数据JSON报文
	 * @param jstr 接收到服务器推送的JSON消息,utf8编码
	 * @param sizejstr 消息长度.
	 * @param param 设置回调时压入的参数.
	 * @remark 高级用户想自己解析服务器推送的快照和SOE消息才需要安装这个回调函数。安装此回调后，上面三个回调cb_OnSscPutSoes，cb_OnPushValSnaps，cb_OnPushObjSnaps
	 * 就不需要安装了，如果同时安装会同时生效被同时回调。不要在这个回调函数例调用其他的需要和实时库交互的接口函数。 
	 */
	static void cb_OnPushJsonMsgs(const char* jstr, int sizejstr, void* param)
	{
		cDemo* pcls = (cDemo*)param;
		pcls->_plog->out(rdbc::logger::logv::dbg, "cb_OnPushJsonMsgs:\n%.*s", sizejstr < 4000 ? sizejstr : 4000, jstr); //输出前4000字符到日志演示以下，
		//实际应该解析JSON消息，根据协议处理。
	}

public:

	/**
	 * @brief 创建实时库句柄，设置参数，开启异步连接
	 * @param surl 实时库连接ulr, "wss://kipway.net"
	 * @param suser 用户名 "opt1"
	 * @param spasswd 密码 "opt1"
	 * @return 0:OK; -1:error;
	 */
	int Open(const char* surl, const char* suser, const char* spasswd)
	{
		if (-1 != _dbh) {
			return 0;
		}
		if (!surl || !*surl || !suser || !*suser || !spasswd || !*spasswd) {
			_plog->out(rdbc::logger::logv::err, "error args");
			return -1; //参数错误
		}
		_dbh = rdb_create();
		if (_dbh < 0) {
			_plog->out(rdbc::logger::logv::err, "rdb_create failed.");
			return -1;
		}
		rdb_soesubscription_setfun(_dbh, cb_OnSscPutSoes, this); //设置SOE处理回调,订阅SOE才需要
		rdb_setmessagenotify(_dbh, cd_onmessage, this);//设置消息回调，用于异步连接。
		rdb_subscription_setfun(_dbh, cb_OnPushValSnaps, this, cb_OnPushObjSnaps, this); //设置订阅标签快照推送接收处理
		rdb_pushjsonmessage_setfun(_dbh, cb_OnPushJsonMsgs, this);//高级用户，可选设置，自己处理服务器推送消息。

		int ncode = rdb_connectex(_dbh, surl, suser, spasswd, 0);//最后再异步连接
		if (SE_OK != ncode) {
			_plog->out(rdbc::logger::logv::err, "connect %s failed. errcode = %d", surl, ncode);
			return -1;
		}
		_wsrul = surl; //保存供以后显示用。
		_username = suser;
		_passwd = spasswd;
		return 0;
	}

	/**
	 * @brief 关闭,应用退出时调用,断开连接，终止运行时线程和释放其他资源。
	 * @return 0
	 */
	int Close()
	{
		if (-1 == _dbh)
			return 0;
		rdb_disconnect(_dbh); //断开连接
		rdb_destory(_dbh);//销毁句柄
		_dbh = -1;//设置句柄无效，避免重复销毁
		return 0;
	}

	/**
	 * @brief 获取绘图数据,使用rdb_valquery获取一条，获取多条曲线数据请使用 getPlotDataX()里的rdb_callmsg()接口直接读取
	 * @return 0:OK; 其他为错误码
	 */
	int getPlotData() {
		if (_status) {
			_plog->out(rdbc::logger::logv::err, "rdb %s is offline", _wsrul.c_str());
			return SE_NOTCONNECT;
		}
		//北京时间2024-8-1一天的绘图数据.
		const char* timeStart = "2024-8-1T0:0:0.000+08:00";
		const char* timeEnd = "2024-8-2T0:0:0.000+08:00";

		std::vector<rec_val> datas;//存储绘图记录的集合
		datas.reserve(2000);
		int nret = rdb_valquery(_dbh, "d0.f01.pv", nullptr,
			rdb_string2jtime(timeStart, strlen(timeStart)) / 100ll, //转换为毫秒。在除以100就是实时库的时标。
			rdb_string2jtime(timeEnd, strlen(timeEnd)) / 100ll,
			0,//提供了开始结束时间，lds无效。 内部版本5114开始 ,参见《rdb_data_exchange_protocol.pdf》 3.2查询值标签历史
			1, //质量无关的绘图数据，可选值 1,3
			[](rec_val pvals[], int nitems, void* pParam) {
				std::vector<rec_val>* pv = (std::vector<rec_val>*)pParam;
				pv->insert(pv->end(), pvals, pvals + nitems);
				return false; //注意读取绘图数据这里返回false,因为绘图数据一次读完了。
			},
			&datas
		);
		if (SE_OK != nret) {
			_plog->out(rdbc::logger::logv::err, "rdb_valquery failed, errcode = %d", nret);
			return nret;
		}
		_plog->out(rdbc::logger::logv::inf, "rdb_valquery success, number records = %zu", datas.size());
		if (!datas.empty()) {
			_plog->out(rdbc::logger::logv::dbg, "\n\ttime\t\t\t QA\t datatype\tvaluse");
			char stime[48] = { 0 };
			std::string sv;
			for (auto& i : datas) { //逐条输出到日志
				rdbc::recValtoString(i, sv);
				rdb_jtime2string(i.time * 100ll, stime, sizeof(stime), 1);
				_plog->append(rdbc::logger::logv::dbg, "%s\t %d\t %s(%d)\t%s\n",
					stime,
					i.cqa,
					rdbc::typeStr(i.cvt), i.cvt,
					sv.c_str()
				);
			}
		}
		return SE_OK;
	}

	/**
	 * @brief 读取历史值。
	 * @param interval_sec 插值间隔秒, 0表示样本值(归档值)
	 * @param flag 标识，可选 0，2，4，5，6，7参见《rdb_data_exchange_protocol.pdf》3.2查询值标签历史
	 * @return 0:OK; 其他为错误码
	 */
	int getHis(int interval_sec, int flag) {
		if (_status) {
			_plog->out(rdbc::logger::logv::err, "rdb %s is offline", _wsrul.c_str());
			return SE_NOTCONNECT;
		}
		//北京时间2024-8-1一天.
		const char* timeStart = "2024-8-1T0:0:0.000+08:00";
		const char* timeEnd = "2024-8-2T0:0:0.000+08:00";

		std::vector<rec_val> datas;//存储结果记录的集合
		datas.reserve(2000);
		int nret = rdb_valquery(_dbh, "d0.f01.pv", nullptr,
			rdb_string2jtime(timeStart, strlen(timeStart)) / 100ll, //转换为毫秒。在除以100就是实时库的时标。
			rdb_string2jtime(timeEnd, strlen(timeEnd)) / 100ll,
			interval_sec * 10,//间隔秒转为为100毫秒数
			flag, //可选值 0，2，4，5，6，7,
			[](rec_val pvals[], int nitems, void* pParam) {
				std::vector<rec_val>* pv = (std::vector<rec_val>*)pParam;
				pv->insert(pv->end(), pvals, pvals + nitems);
				return true; //注意读取非绘图数据这里返回true,表示继续读到结束时间止。
			},
			&datas
		);
		if (SE_OK != nret) {
			_plog->out(rdbc::logger::logv::err, "rdb_valquery failed, errcode = %d", nret);
			return nret;
		}
		_plog->out(rdbc::logger::logv::inf, "rdb_valquery success, number records = %zu", datas.size());
		if (!datas.empty()) {
			_plog->out(rdbc::logger::logv::dbg, "\n\ttime\t\t\t QA\t datatype\tvaluse");
			char stime[48] = { 0 };
			std::string sv;
			for (auto& i : datas) { //逐条输出到日志
				rdbc::recValtoString(i, sv);
				rdb_jtime2string(i.time * 100ll, stime, sizeof(stime), 1);
				_plog->append(rdbc::logger::logv::dbg, "%s\t %d\t %s(%d)\t%s\n",
					stime,
					i.cqa,
					rdbc::typeStr(i.cvt), i.cvt,
					sv.c_str()
				);
			}
		}
		return SE_OK;
	}

	/**
	 * @brief 直接使用通信协议请求应答。适合高级用户和rdbapi动态库未提供的命令请求。
	 * @param jsReq 请求的json命令，utf8编码，符合JSON规范RFC8259。
	 * @param sizejsReq 请求的json命令长度。
	 * @param jsResp 应答的json消息，客户端需要自己做转义和解析为数据对象。
	 * @return 0：OK; 其他为错误码
	 * @remark 适合根据《rdb_data_exchange_protocol.pdf》协议直接和实时库交互的高级用户使用。
	 * 这里的rdb_callmsg()是需要等待实时库应答的，等待超时为60秒；如果不需要处理应答，或者没有应答，请使用rdb_putmsg()推送接口函数。
	 */
	int Call(const char* jsReq, size_t sizejsReq, std::string& jsResp)
	{
		if (_status) {
			_plog->out(rdbc::logger::logv::err, "rdb %s is offline", _wsrul.c_str());
			return SE_NOTCONNECT;
		}
		_plog->out(rdbc::logger::logv::dbg, "request:\n%.*s", (int)sizejsReq, jsReq);//输出请求消息日志。
		jsResp.clear();
		int nret = rdb_callmsg(_dbh, jsReq, (int)sizejsReq, [](const char* response, int sizeresponse, void* app_param) {
			std::string* ps = (std::string*)app_param;
			ps->clear();
			ps->append(response, sizeresponse);
			},
			(std::string*)&jsResp
		);
		if (SE_OK != nret) {
			_plog->out(rdbc::logger::logv::err, "Call failed, errcode = %d", nret);
			return nret;
		}
		else { //输出原始的应答消息到日志。
			_plog->out(rdbc::logger::logv::dbg, "response:\n%.*s", (int)jsResp.size(), jsResp.c_str());
		}
		return nret;
	}

	/**
	 * @brief 使用rdb_callmsg获取实时库运行信息
	 * @return 0:OK; 其他为错误码
	 */
	int rdbInfo()
	{
		if (_status) {
			_plog->out(rdbc::logger::logv::err, "rdb %s is offline", _wsrul.empty() ? "" : _wsrul.c_str());
			return SE_NOTCONNECT;
		}
		std::string jsReq = "{\"request\":\"rdb_dbinfo\",\"seqno\":";
		jsReq.append(std::to_string(NextSeqNo())).push_back('}');
		std::string jsResp;
		jsResp.reserve(1000);
		return Call(jsReq.c_str(), jsReq.size(), jsResp); //如果成功，结果在jsResp里
	}

	/**
	 * @brief 使用rdb_callmsg获取绘图数据多条
	 * @return 0：OK; 其他为错误码
	 */
	int getPlotDataX()
	{
		if (_status) {
			_plog->out(rdbc::logger::logv::err, "rdb %s is offline", _wsrul.c_str());
			return SE_NOTCONNECT;
		}

		//北京时间2024-8-1一天的绘图数据.
		const char* timeStart = "2024-8-1T0:0:0.000+08:00";
		const char* timeEnd = "2024-8-2T0:0:0.000+08:00";
		//注意如果读取实时曲线，结束时间设置为当前时间，开始时间根据曲线的细节需要减10分钟，20分钟，或自定义的区间

		std::vector<const char*> tags;
		tags.push_back("d0.f01.pv");//注意如果是非utf8编码，先做转码，如果含有需要转义的特殊字符，还要做转义。
		tags.push_back("d0.k01.pv");//加第二条曲线。

		std::string jsReq;
		jsReq.reserve(500);

		//组织rdb_plotdata请求命令,只需要开始时间，结束时间(内部版本5114开始)和标签，
		jsReq.push_back('{');
		jsReq.append("\"request\":\"rdb_plotdata\",").append("\"seqno\":").append(std::to_string(NextSeqNo()));

		jsReq.append(",\"tags\":"); //添加标签数组字段
		char cp = '[';
		for (auto i : tags) {
			jsReq.push_back(cp);
			if (cp != ',')
				cp = ',';
			jsReq.append("\"").append(i).push_back('"');
		}
		jsReq.push_back(']');

		jsReq.append(",\"time_begin\":\"").append(timeStart).push_back('"'); //开始时间
		jsReq.append(",\"time_end\":\"").append(timeEnd).push_back('"');//结束时间,
		jsReq.append(",\"flag\":1");//flag=1 默认绘图数据,QA=1的shutdown的时间段数据不插值
		jsReq.push_back('}');

		std::string jsResp;
		jsResp.reserve(1024 * 20);//优化效率预设值20K空间
		return Call(jsReq.data(), jsReq.size(), jsResp);
	}

	/**
	 * @brief 读取快照
	 * @return 0:OK; 其他为错误码
	 */
	int getSnapshot()
	{
		if (_status) {
			_plog->out(rdbc::logger::logv::err, "rdb %s is offline", _wsrul.c_str());
			return SE_NOTCONNECT;
		}
		std::vector<const char*> tags; //演示一次性读取4个值标签和1个字符串标签。
		tags.reserve(5);
		tags.push_back("d0.f01.pv");
		tags.push_back("d0.dbl01.pv");
		tags.push_back("d0.i01.pv");
		tags.push_back("d0.l01.pv");
		tags.push_back("d0.str01.pv");

		std::vector<rec_tagval> vals; //存放值标签结果集的集合
		vals.reserve(4);
		std::vector<rec_tagobj> objs;//存放字符串标签结果集的集合
		objs.reserve(1);

		int nret = rdb_getsnapshot(_dbh, tags.data(), (int)tags.size(),
			[](const rec_tagval vals[], size_t valsize, void* param) {
				std::vector<rec_tagval>* pv = (std::vector<rec_tagval>*)param;
				pv->insert(pv->end(), vals, vals + valsize);
			}, &vals,
			[](const rec_tagobj objs[], size_t objsize, void* param) {
				std::vector<rec_tagobj>* pv = (std::vector<rec_tagobj>*)param;
				pv->insert(pv->end(), objs, objs + objsize);
			}, &objs
		);
		if (SE_OK != nret) {
			_plog->out(rdbc::logger::logv::err, "rdb_getsnapshot failed, errcode = %d", nret);
			return nret;
		}
		//输出到日志
		_plog->out(rdbc::logger::logv::dbg, "rdb_getsnapshot success, numreords = %zu", vals.size() + objs.size());
		_plog->out(rdbc::logger::logv::dbg, "\ntagname\t\t\t\ttime\t\t\t QA\t datatype\tvaluse");
		std::string sv;
		char stime[48] = { 0 };
		for (auto& i : vals) { //输出值标签
			rdbc::recValtoString(i.val, sv);
			rdb_jtime2string(i.val.time * 100ll, stime, sizeof(stime), 1);
			_plog->append(rdbc::logger::logv::dbg, "%s\t = %s\t %d\t %s(%d)\t%s\n",
				i.sname,
				stime,
				i.val.cqa,
				rdbc::typeStr(i.val.cvt), i.val.cvt,
				sv.c_str()
			);
		}

		for (auto& i : objs) { //输出字符串类型标签
			rdb_jtime2string(i.var.time * 100ll, stime, sizeof(stime), 1);
			_plog->append(rdbc::logger::logv::dbg, "%s\t = %s\t %d\t %s(%d)\t%s\n",
				i.sname,
				stime,
				i.var.cqa,
				rdbc::typeStr(i.var.cvt), i.var.cvt,
				i.var.sdata
			);
		}
		return nret;
	}

	/**
	 * @brief 读取实时库内部版本
	 * @return 返回值大于5000为内部版本号, -1为失败； 5000表示实时库版本太旧，不支持获取内部版本号。
	 */
	int dbver() {
		if (_status) {
			_plog->out(rdbc::logger::logv::err, "rdb %s is offline", _wsrul.c_str());
			return -1;
		}
		int nver = rdb_curdb_inver(_dbh);
		_plog->out(rdbc::logger::logv::inf, "inner version = %d", nver);
		return nver;
	}

	/**
	 * @brief 测试写快照
	 * @return 0：OK; 其他为错误码
	 */
	int tstWriteSnaps()
	{
		if (_status) {
			_plog->out(rdbc::logger::logv::err, "rdb %s is offline", _wsrul.c_str());
			return SE_NOTCONNECT;
		}
		std::vector<rec_tagval> vals;
		rec_tagval v;
		memset(&v, 0, sizeof(v)); //清零包括时标 ，下面也不填时标表示使用实时库时标

		strcpy(v.sname, "d0.f01.pv");
		v.val.cvt = DT_FLOAT32;
		v.val.f32 = (float)1.5;
		v.val.cqa = QA_OK;
		vals.push_back(v);

		strcpy(v.sname, "d0.k01.pv");
		v.val.cvt = DT_DIGITAL;
		v.val.i32 = 1;
		v.val.cqa = QA_OK;
		vals.push_back(v);

		strcpy(v.sname, "d1.k01.pv");
		v.val.cvt = DT_DIGITAL;
		v.val.i32 = 1;
		v.val.cqa = QA_OK;
		vals.push_back(v);

		int nr = rdb_valputsnapshot(_dbh, vals.data(), (int)vals.size());
		if (SE_OK != nr) {
			_plog->out(rdbc::logger::logv::err, "rdb_valputsnapshot failed. error %d %s", nr, rdb_geterrmsg(nr));
		}
		else {
			for (auto& i : vals) {//检查小错误码
				if (i.val.cerr) {
					_plog->out(rdbc::logger::logv::err, "rdb_valputsnapshot tag %s failed. error %d", i.sname, i.val.cerr);
				}
			}
		}
		return nr;
	}

	/**
	 * @brief 测试写字符串标签快照
	 * @return 0：OK; 其他为错误码
	 */
	int tstWriteObjectSnaps()
	{
		if (_status) {
			_plog->out(rdbc::logger::logv::err, "rdb %s is offline", _wsrul.c_str());
			return SE_NOTCONNECT;
		}
		std::vector<rec_tagobj> objs;
		rec_tagobj v;
		memset(&v, 0, sizeof(v)); //清零包括时标 ，下面也不填时标表示使用实时库时标

		strcpy(v.sname, "d0.str01.pv");
		v.var.cvt = DT_STRING;
		v.var.cqa = QA_OK;
		v.var.uslen = sprintf(v.var.sdata, "s1:%jd", (int64_t)::time(nullptr));
		objs.push_back(v);

		strcpy(v.sname, "d0.str02.pv");
		v.var.cvt = DT_STRING;
		v.var.cqa = QA_OK;
		v.var.uslen = sprintf(v.var.sdata, "s2:%jd", (int64_t)::time(nullptr));
		objs.push_back(v);

		int nr = rdb_objputsnapshot(_dbh, objs.data(), (int)objs.size());
		if (SE_OK != nr) {
			_plog->out(rdbc::logger::logv::err, "rdb_objputsnapshot failed. error %d %s", nr, rdb_geterrmsg(nr));
		}
		else {
			for (auto& i : objs) {//检查小错误码
				if (i.var.cerr) {
					_plog->out(rdbc::logger::logv::err, "rdb_objputsnapshot tag %s failed. error %d", i.sname, i.var.cerr);
				}
			}
		}
		return nr;
	}

	int tstSubscribe() //订阅测试
	{
		if (_status) {
			_plog->out(rdbc::logger::logv::err, "rdb %s is offline", _wsrul.c_str());
			return SE_NOTCONNECT;
		}
		std::vector<const char*> tags{ "d0.f0*","d1.str01.pv","d1.obj01.pv" };
		return rdb_sscsnaps(_dbh, 1, 1, tags.data(), (int)tags.size(), [](const char* tagname, int errcode, void* param) {
			cDemo* pcls = (cDemo*)param;
			pcls->_plog->out(rdbc::logger::logv::err, "subscribe %s error %d", tagname, errcode);
			}, this);		
	}
};

const char* shelp = "rdb CPP demo，order list:\n\n"
"  help\t\t: 帮助-view this information\n"
"  exit\t\t: 退出-exit this demo applacation\n"
"  dbver\t\t: 实时库版本-get database inner version\n"
"  dbinfo\t: 实时库运行信息-get database information\n"
"  snaps\t\t: 读取快照-get tags value snapshots\n"
"  plot\t\t: 绘图数据-get one tag plot value records\n"
"  plotx\t\t: 高级模式同时读取多条绘图数据-get tags plot value records\n"
"  hisorg\t: 读取归档值(样本值)-get history archived records\n"
"  hismin\t: 读取分钟数据-get history minute value records\n"
"  hishour\t: 读取小时数据-get history hour value records\n"
"  hishourmin\t: 读取小时最小值-get history hour minimum value records\n"
"  hishourmax\t: 读取小时最大值-get history hour maximum value records\n"
"  writesnaps\t: 写快照例子-write snapshots\n"
"  subscribe\t: 订阅标签快照-subscribe tags snapshot\n"
"\n"
;

/**
 * @brief 入口函数
 * @param argc 4
 * @param argv url username passwd
 * @return 0
 */
int main(int argc, const char* argv[])
{
#ifdef _WIN32
	SetConsoleOutputCP(65001); //设置控制台输出字符集utf8编码，和linux默认字符集一致。
#endif
	if (4 != argc) {
		printf("usage:\n  rdbcppdemo url username passwd\ndemo:\n  rdbcppdemo wss://kipway.net opt1 opt1\n");
		return 0;
	}
	cDemo demo;
	if (demo.Open(argv[1], argv[2], argv[3])) {
		return 0;
	} //rdbapi是自动重连的,Open后就不管了，连接登录成功会在demo::cd_onmessage()里通知。

	printf("%s", shelp);

	char sod[200] = { 0 }; //命令行
	char* pc;
	while (1) {
		if (!fgets(sod, sizeof(sod), stdin))
			continue;
		pc = (char*)sod;
		while (*pc) { //去掉linux的行尾\n
			if (*pc == '\n' || *pc == '\r') {
				*pc = '\0';
				break;
			}
			++pc;
		}
		if (!*sod)
			continue;
		if (!stricmp("exit", sod)) {
			break;
		}
		else if (!stricmp("help", sod)) { //读取快照
			printf("%s", shelp);
		}
		else if (!stricmp("plot", sod)) { //获取一条绘图数据
			demo.getPlotData();
		}
		else if (!stricmp("plotx", sod)) { //获取多条绘图数据,使用rdb_callmsg直接和实时库交互。
			demo.getPlotDataX();
		}
		else if (!stricmp("snaps", sod)) { //读取快照
			demo.getSnapshot();
		}
		else if (!stricmp("hismin", sod)) { //获取分钟值
			demo.getHis(60, 0);
		}
		else if (!stricmp("hishour", sod)) { //获取小时值
			demo.getHis(3600, 0);
		}
		else if (!stricmp("hishourmin", sod)) { //获取小时最小值, inver 5114及以后
			demo.getHis(3600, 4);
		}
		else if (!stricmp("hishourmax", sod)) { //获取小时最大值, inver 5114及以后
			demo.getHis(3600, 5);
		}
		else if (!stricmp("hisorg", sod)) { //获取样本值(实时库中实际落地归档的记录)
			demo.getHis(0, 0);
		}
		else if (!stricmp("dbinfo", sod) || !stricmp("rdbinfo", sod)) { //获取实时库运行信息
			demo.rdbInfo();
		}
		else if (!stricmp("dbver", sod) || !stricmp("rdbver", sod)) { //读取实时库内部版本号.
			demo.dbver();
		}
		else if (!stricmp("writesnaps", sod)) { //写快照测试.
			demo.tstWriteSnaps();
			demo.tstWriteObjectSnaps();
		}
		else if (!stricmp("subscribe", sod)) {
			demo.tstSubscribe();
		}
		else {
			printf("unkown order!\n");
			printf("%s", shelp);
		}
	}
	demo.Close();
	return 0;
}
