﻿/*!
\file dacdrv.h
\brief 数据采集驱动接口定义

\author	 jiangyong
\version 5.0
\update  2016-5-7
*/

#ifndef DAC_DRV_H
#define DAC_DRV_H

#include "rdbdef.h"

#define DACRET_ERRHANDLE (-1) //错误的句柄
#define DACRET_OK        0 //成功
#define DACRET_FAILED    1 //失败
#define DACRET_ERRARGS   2 //参数错误
#define DACRET_NOTAG     3 //无此标签
#define DACRET_TAGACCESS 4 //标签访问错误，比如不可写标签
#define DACRET_CTRLIOERR 5 //设备输出IO错误
#define DACRET_CTRLDEV   6 //设备操作失败

#define LOGLEVEL_MSG     0 //日志日志
#define LOGLEVEL_WRN     1 //警告日志
#define LOGLEVEL_ERR     2 //错误日志
#define LOGLEVEL_DBG     3 //调试日志

#define DAC_R            1 //只读
#define DAC_W            2 //只写
#define DAC_RW           3 //读写

#ifdef _WIN32
#   if defined(__cplusplus)
#       define DACDRV_FUNC extern "C" __declspec(dllexport)
#   else
#       define DACDRV_FUNC __declspec(dllexport)
#   endif
#else
#   define DACDRV_FUNC extern "C"
#endif

/*!
\breif 消息回调
*/
typedef void(*dac_onmsg)(const char* sdevicename, int nmsgtype, const char* smsg, bool blog, void* param);

/*!
\brief 值数据回调
*/
typedef void(*dac_ontagval)(rec_tagval pvs[], int nsize, void *param);

/*!
\brief 对象数据回调
*/
typedef void(*dac_ontagobj)(rec_tagobj pvs[], int nsize, void *param);

/*!
\brief SOE数据回调
*/
typedef void(*dac_onsoe)(rec_soe pvs[], int nsize, void *param);

/*!
\brief 标签属性回调
\return 返回0表示终止标签读取,非零表示继续
*/
typedef bool(*dac_ontag)(const char* srdbtagname, int ntype, int naccess, void *param);

/*!
\brief 标签属性回调扩展,增加单位和描述
\return 返回0表示终止标签读取,非零表示继续
*/
typedef bool(*dac_ontagex)(const char* srdbtagname, int ntype, int naccess, const char* sunit, const char* sdes, void *param);


/*!
\brief 获取版本
\return 返回驱动规范的版本。
\remark 为了编译支持以后的规范升级设计的，目前固定返回500表示 Ver5.00
*/
DACDRV_FUNC int dac_getversion();

/*!
\brief  创建一个数据采集驱动对象
\param  sdevicename [in] 设备名
\param  scfgfile [in] 配置文件名，全路径
\param  onmsg [in] 消息信息和日志回调函数
\param  onmsgparam [in] 消息信息和日志回调函数的param参数
\param  dac_ontagval [in] 值标签数据到达通知回调函数
\param  ontagvalparam [in] 值标签数据到达通知回调函数的参数
\param  ontagobj [in] 对象标签数据到达通知回调函数
\param  ontagobjparam [in] 对象标签数据到达通知回调函数的参数
\param  onsoe [in] SOE事件到达通知回调函数
\param  ontagobjparam [in] SOE事件到达通知回调函数的参数

\return 返回驱动对象句柄（大于等于0的整数）,如果失败，返回-1。
*/
DACDRV_FUNC int dac_create(const char* sdevicename, const char* scfgfile,
	dac_onmsg onmsg, void* onmsgparam,
	dac_ontagval ontagval, void* ontagvalparam,
	dac_ontagobj ontagobj, void *ontagobjparam,
	dac_onsoe onsoe, void* onsoeparam
);

/*!
\brief 删除驱动对象
\param h [in] dac_create创建的句柄
*/
DACDRV_FUNC void dac_destroy(int h);

/*!
\brief 读取所有标签
\param h dac_create创建的句柄
\param ontag 标签获取回掉函数
\param ontagparam [in] ontag的参数
\return 返回0表示成功,否则为错误码
*/
DACDRV_FUNC int dac_getalltags(int h, dac_ontag ontag, void* ontagparam);

/*!
\brief 读取所有标签扩展版
\param h dac_create创建的句柄
\param ontag 标签获取回掉函数
\param ontagparam [in] ontag的参数
\return 返回0表示成功,否则为错误码
*/
DACDRV_FUNC int dac_getalltagsex(int h, dac_ontagex ontag, void* ontagparam);

/*!
\brief 启动
\param h dac_create创建的句柄
\return 返回0表示成功,否则为错误码
*/
DACDRV_FUNC int dac_start(int h);

/*!
\brief 启动
\param h dac_create创建的句柄
\return 返回0表示成功,否则为错误码
*/
DACDRV_FUNC int dac_stop(int h);

/*!
\brief 是否在运行
\param h dac_create创建的句柄
\return 0表示没有运行，否则为正在运行
*/
DACDRV_FUNC bool dac_isrun(int h);

/*!
\brief 写设备值记录，输出控制接口
\param h dac_create创建的句柄
\param pval 标签值记录
\return 返回0表示成功,否则为错误码
\remark 这是输出控制接口函数，pval中可以存储TD_DIGITAL,DT_INT32,DT_FLOAT32,DT_INT64，DT_FLOAT64类型
*/
DACDRV_FUNC int dac_devicewriteval(int h, rec_tagval *pval);

/*!
\brief 写设备对象记录，输出控制接口
\param h dac_create创建的句柄
\param pobj 标签对象记录
\return 返回0表示成功,否则为错误码
\remark 这是输出控制接口函数，pobj中可以是DT_STRINGT或DT_OBJECT类型
*/
DACDRV_FUNC int dac_devicewriteobj(int h, rec_tagobj *pobj);

/*!
\brief 读取设备信息
\param h dac_create创建的句柄
\param sout [out]输出缓冲区
\param poutsize [in/out] 输出缓冲区大小和实际字节数,调用前填写sout的大小，返回时填写需要输出的信息字节数，如果sout小于输出的字节数+1，则不会填写sout
\return 返回0表示成功,否则为错误码,
\remark 返回时注意检查poutsize回填的值，小于sout提供的大小才会输出到sout。
*/
DACDRV_FUNC int dac_getinfo(int h, char *sout, size_t *poutsize);

#endif// DAC_DRV_H
