﻿/*!
\file real-time database define.h
\brief 实时库数据定义，用于客户端接口和前段数据采集

\update  2020-8-1   增加ctagtype字段导入
\update  2020-6-14  增加人工置数
\update  2018-8-23  增加TA_COMPTIMER和TA_COMPEXTIMER两种压缩方式

*/
#ifndef RDBDEF_H
#define RDBDEF_H

typedef char				T_I8; //1字节整数
typedef unsigned char		T_U8; //1字节无符号整数

typedef short int			T_I16;//2字节整数
typedef unsigned short int	T_U16;//2字节无符号整数

typedef int					T_I32;//4字节整数
typedef unsigned int		T_U32;//4字节无符号整数

typedef long long			T_I64;//8字节整数
typedef unsigned long long	T_U64;//8字节无符号整数

typedef float				T_F32;//4字节浮点数
typedef double				T_F64;//8字节双精度浮点数

typedef	T_I64				T_TIME;//64位时标，精确到100毫秒，UTC/GMT(国际时间)自1970-1-1开始的100毫秒数 = time_t * 10

//数据类型
#define DT_VOID			    0	//空类型
#define DT_DIGITAL		    1	//开关量,也是4字节整数
#define DT_INT32		    2	//4字节整数
#define DT_FLOAT32		    3	//4字节浮点数
#define DT_INT64		    4	//8字节整数
#define DT_FLOAT64		    5	//8字节浮点数
#define DT_STRING		    6	//字符串
#define DT_OBJECT		    7	//对象类型

//标签分类
#define TGCLS_DEC		    0	//现场设备标签
#define TGCLS_DEFINE	    1	//定义标签
#define TGCLS_PRESET	    2	//预处理标签
#define TGCLS_CURVE		    3	//曲线标签
#define TGCLS_SYSTAG	    4	//系统标签

//压缩方式
#define TA_COMPNO		    0	//不压缩
#define TA_COMPPER		    1	//百分比精度压缩
#define TA_COMPVAL		    2	//绝对值精度压缩
#define TA_COMPTIMER        3   //定时存储,使用最大压缩周期间隔存储
#define TA_COMPEXTIMER      4   //增强定时存储,使用最大压缩周期间隔存储,并存储区间内的最大最小值记录

//数据质量
#define QA_OK			    0	//质量
#define QA_SHUTDOWN		    1   //关机,数据不可靠
#define QA_ERRDATA		    2	//数据无效
#define QA_NOTAG		    3	//无此标签
#define QA_ERRTIME		    4	//错误的时标
#define QA_MANUAL 			5	//人工置数

//权限定义
#define OPT_POW_READ	    0x01  //读取数据
#define OPT_POW_WRITE	    0x02  //写数据
#define OPT_POW_WTAG	    0x04  //写标签
#define OPT_POW_MAN         0x08  //账号管理
#define OPT_POW_CTRL	    0x10  //控制权限

//标签导入策略,某位有效表示需要修改该项
#define TAG_MASK_DES		0x0001	//描述
#define TAG_MASK_UNIT		0x0002	//单位
#define TAG_MASK_TYPE		0x0004  //标签类型,ctagtype字段，内部暂未使用
#define TAG_MASK_COMPEXC	0x0008	//压缩属性
#define TAG_MASK_CLASS  	0x0010	//标签分类,cclass字段
#define TAG_MASK_ARCHIVE	0x0020	//归档
#define TAG_MASK_VDIGITS	0x0040	//显示精度
#define TAG_MASK_STEP		0x0080	//梯形插值
#define TAG_MASK_LIMIT		0x0100	//上下限值
#define TAG_MASK_ALARM		0x0400	//越限报警设置

//标签导入结果
#define TAGIN_NOTMODIFY		0		//标签未修改
#define TAGIN_MODIFY		1		//标签修改
#define TAGIN_ADD			2		//标签添加

//rdb_valgetsection中参数nflag的定义
#define SECTION_INSERT      0       //插值
#define SECTION_AFTER		1		//之后, 时标 >= 指定时间
#define SECTION_BEFORE		(-1)	//之前, 时标 <= 指定时间

#define SE_OPT_OK			0		//成功
#define SE_OPT_NOACTOR		0x20	//无此角色
#define SE_OPT_NOUSER		0x21	//无此账号
#define SE_OPT_PASSERR		0x22	//密码错误
#define SE_OPT_NOPOWER		0x23	//无此权限
#define SE_OPT_USEREXIST	0x24	//账号重复
#define SE_OPT_ACTOREXIST	0x25	//角色重复
#define SE_OPT_ACTORUSE		0x26	//角色正在使用
#define SE_OPT_NOLOGIN		0x27	//没有登录
#define SE_OPT_NOTACTIVE	0x28	//账号冻结
#define SE_OPT_OPTFULL		0x29	//账号满
#define SE_OPT_ACTFULL		0x2A	//角色满
#define SE_OPT_NOTMODPSW	0x2B	//不容许修改密码

#define SE_TG_NOTAG			0x30	//无此标签
#define SE_TG_TAGFULL		0x31	//标签满
#define SE_TG_NOARCHIVE		0x32	//不归档
#define SE_TG_MODSYSTAG 	0x33	//不能修改系统标签

#define SE_DA_TIME			0x40	//时标错误
#define SE_DA_TYPE			0x41	//类型错误
#define SE_DA_QA			0x42	//质量错误
#define SE_DA_NODATA		0x43	//无数据

#define SE_CTRLNOROUTE		0x61	//!<无路由不能写
#define SE_CTRLOUTERR       0x62    //!<输出错误

#define SE_SRV_FRMNOOD		0xFF	//无此命令
#define SE_SRV_FRMERR		0xFE	//报文错误
#define SE_SRV_DBVER		0xFD	//API和实时库版本不一致
#define SE_SRV_BACKUOPSRV   0XFB    //备份服务器
#define SE_SRV_DEMOOVER		0xFC	//试用过期,历史数据服务停止


//标签越限配置和越限快照读取中cerr定义的值
#define LIMIT_LOW		1
#define LIMIT_HIGH		2
#define LIMIT_LOWLOW	4
#define LIMIT_HIGHHIGH	8

/*!
\brief 标签属性记录
*/
struct rec_tag
{
	T_U32	uid;		//标签ID

	T_I8    cdatatype;	//数据类型
	T_I8    ctagtype;	//标签类型,没有使用,填写TAGTYPE_NONE,可有客户端自定义。
	T_I8    ccomptype;	//压缩类型
	T_I8   	cres;		//保留

	T_F32   fcompval;	//压缩精度
	T_I8    sname[80];	//标签名
	T_I8    sdes[80];	//描述
	T_I8    sunit[16];	//工程单位

	T_I8    cdigits;	//小数位数，-20  到  10  ; >0 小数位数，小于0，有效位数
	T_I8    cclass;		//分类,默认TGCLS_DEC
	T_I8    cstep;		//梯形插值,1梯形，0线性
	T_I8    carchive;	//存盘,1存盘，0不存

	T_F32   fdnval;		//下限
	T_F32   fupval;		//上限

	T_F32   fexcdev;	//列外偏差,如果是百分比，则和百分比压缩意义相同
	T_I16   snexcmin;	//最小列外偏差时间
	T_I16   snexcmax;	//最大列外偏差时间

	T_I16   sncompmin;	//最小压缩时间间隔(单位秒)
	T_I16   sncompmax;	//最大压缩时间间隔(单位秒)

	T_I32  alarmtype;   //报警状态
	float  alarm_llv;   //报警下下限
	float  alarm_lv;	//报警下限
	float  alarm_hv;	//报警上限
	float  alarm_hhv;	//报警上上限

	T_U8	sres[24];	//保留,清零
};		//sizeof() = 256


/*!
\brief 值数据记录
*/
struct rec_val
{
	T_TIME	time;		//GMT时标,从1970年1月1日开始的100毫秒数
	union
	{
		T_I32  	i32;	//存储didital、int32
		T_F32  	f32;	//存储float32
		T_I64	i64;	//存储int64
		T_F64 	f64;	//存储float64
		T_U64	u64;	//存储64位整数
	};
	T_I8	cvt;		//类型
	T_I8	cqa;		//质量
	T_I8	cerr;		//错误码
	T_I8	cres;		//保留,读取越限标签快照时，cres为越限状态,LIMIT_LOW,LIMIT_HIGH,LIMIT_LOWLOW,LIMIT_HIGHHIGH
	T_U32	unres;		//保留
};  //sizeof() = 24

/*!
\brief值标签数据记录
*/
struct rec_tagval
{
	char sname[80];     //标签名
	rec_val val;        //值记录
}; // sizeof = 104

#define MAX_OBJ_LEN 1000    //最大对象数据字节数

/*!
\brief对象数据记录
*/
struct  rec_obj {
	T_TIME	time;			//时标

	T_U32	uobjtype;		//类型,用户自定义,可当检索条件
	T_U32   ures;			//以后扩展用,比如第2索引BATID

	T_I8	cvt;			//数据类型,DT_STRING,DT_BLOB
	T_I8	cqa;			//质量
	T_I8	cerr;			//错误码
	T_I8	cres;			//保留

	T_U16	usres;			//保留
	T_U16	uslen;			//数据长度，即sdata的长度，如果是字符串，则不包含结束符0

	T_I8	sdata[MAX_OBJ_LEN];//!<对象数据
};  //sizeof() = 1024

/*!
\brief 对象标签数据记录
*/
struct rec_tagobj
{
	char sname[80]; //!<标签名
	rec_obj var;    //!<对象记录
}; //sizeof() = 1104

/*!
\brief账号记录
*/
struct rec_operator
{
	char	sname[16];		//用户名<=15字符
	T_U8	smd5pass[16];   //二进制的MD5加密后的密码
	char	sactor[16];		//角色名,内部表关联用
	char	sdes[80];		//描述
	T_I32	lbactive;		//是否活动,0不活动，1活动
	char    nomodpsw;		//禁止修改密码；1：禁止；0：容许
	char	sres[27];		//保留    
};	//sizeof = 160


/*!
\brief角色记录
*/
struct rec_actor
{
	char	sname[16];		//角色名
	char	sdes[80];		//描述
	T_U32	dwpower;		//权限
	char	sres[28];		//保留
};	//sizeof = 128

/*!
\brief事件记录
*/
struct rec_soe
{
	T_TIME	time;       //GMT时标,从1970年1月1日开始的100毫秒数
	T_U32	uautokey;   //自动主键,解决时间相同的SOE事件主键冲突
	T_I32	type;       //事件类型,4字节有符号,(用户自定义,比如可定义越上限，越下限等)
	T_I16	argtype;    //事件参数格式，2字节带符号，<0 为基于字符串的格式，>0为基于2进制的格式。具体有用户定义,用于客户端解析参数。
	T_U16	arglen;	    //参数长度(0-240)，基于字符串格式的长度不包括结尾的0。
	T_I32	level;      //事件级别使用

	char    source[80]; //事件来源，0结束的字符串，最多存储79字符，可填写标签名或者其他内容
	char    sdes[160];  //事件描述，0结束的字符串，最多存储159字节
	char    sarg[240];  //事件参数。根据argtype和arglen填写。
	char    cflag;      //事件已经处理标识
	char    cstatus;    //SOE UPdate返回时用于指示是否更新,0表示更新成功，1表示不存在或更新失败
	char    res[6];     //保留
}; // sizeof() = 512

#endif //RDBDEF_H